<?php

namespace App\Http\Controllers;

use App\Models\Registration;
use App\Mail\RegistrationVoucherMail;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class RegistrationController extends Controller
{
    public function index()
    {
        return view('index');
    }

    public function validateField(Request $request)
    {
        $field = (string) $request->input('field');
        $value = $request->input('value');

        $rules = match ($field) {
            'email'   => ['nullable', 'email'],
            'mobile'  => ['required', 'regex:/^\d{10}$/'],
            'city'    => ['required', 'string', 'max:80'],
            'invoice' => ['required', 'numeric', 'min:0'],
            'terms'   => ['accepted'],
            default   => [],
        };

        if (empty($rules)) return response()->json(['ok' => true], 200);

        $validator = Validator::make([$field => $value], [$field => $rules], [
            'mobile.regex' => 'Enter a valid 10-digit mobile number.',
        ]);

        if ($validator->fails()) {
            return response()->json(['ok' => false, 'errors' => $validator->errors()], 422);
        }

        return response()->json(['ok' => true], 200);
    }

    public function store(Request $request)
    {
        // NOTE: no file/mimes rules here to avoid fileinfo
        $rules = [
            'email'       => ['nullable', 'email'],
            'mobile'      => ['required', 'regex:/^\d{10}$/'],
            'city'        => ['required', 'string', 'max:80'],
            'invoice'     => ['required', 'numeric', 'min:0'],
            'upload_file' => ['required'], // manual checks below
            'terms'       => ['accepted'],
        ];
        $messages = ['mobile.regex' => 'Enter a valid 10-digit mobile number.'];

        $validator = Validator::make($request->all(), $rules, $messages);
        if ($validator->fails()) {
            if ($request->expectsJson() || $request->ajax()) {
                return response()->json(['ok' => false, 'errors' => $validator->errors()], 422);
            }
            return back()->withErrors($validator)->withInput();
        }

        // -------- Manual file checks (no php_fileinfo required) --------
        if (!$request->hasFile('upload_file')) {
            return $this->fileErrorResponse($request, 'upload_file', 'Please upload an image or PDF.');
        }

        $file = $request->file('upload_file');

        if (!$file->isValid()) {
            return $this->fileErrorResponse($request, 'upload_file', 'File upload failed. Try again.');
        }

        // Whitelist by extension
        $allowedExt = ['jpg','jpeg','png','webp','pdf'];
        $ext = strtolower($file->getClientOriginalExtension() ?: '');
        if (!in_array($ext, $allowedExt, true)) {
            return $this->fileErrorResponse($request, 'upload_file', 'Only JPG, PNG, WEBP or PDF allowed.');
        }

        // Size check (8 MB)
        $maxBytes = 8 * 1024 * 1024;
        if (($file->getSize() ?? 0) > $maxBytes) {
            return $this->fileErrorResponse($request, 'upload_file', 'File too large. Max 8 MB allowed.');
        }

        // Optional: light client MIME check (from browser header, not fileinfo)
        $clientMime = strtolower($file->getClientMimeType() ?? '');
        $okClientMime = str_starts_with($clientMime, 'image/') || $clientMime === 'application/pdf';
        if (!$okClientMime) {
            // Not fatal—comment out if you want to allow anything with allowed extension
            return $this->fileErrorResponse($request, 'upload_file', 'Unsupported file type.');
        }

        // Save file directly to public/uploads (no storage:link needed)
        $basename  = 'reg_'.time().'_'.bin2hex(random_bytes(4));
        $filename  = $basename.'.'.$ext;
        $targetDir = public_path('uploads');
        if (!is_dir($targetDir)) { @mkdir($targetDir, 0755, true); }
        $file->move($targetDir, $filename);
        $path = 'uploads/'.$filename;
        // ---------------------------------------------------------------

        // Voucher code
        $voucher = $this->generateUniqueVoucher();

        // Save
        $reg = Registration::create([
            'email'          => $request->input('email') ?: null,
            'mobile'         => $request->input('mobile'),
            'city'           => $request->input('city'),
            'invoice_value'  => $request->input('invoice'),
            'upload_path'    => $path,
            'accepted_terms' => 1,
            'status'         => 'pending',
            'voucher_code'   => $voucher,
        ]);

        // Email (if present)
        if (!empty($reg->email)) {
            try { Mail::to($reg->email)->send(new RegistrationVoucherMail($voucher, null)); }
            catch (\Throwable $e) { /* optional log */ }
        }

        // WhatsApp (hard-coded creds; same as your curl). Remove if not needed.
        try { $this->sendWhatsappTemplateHardcoded($reg->mobile, $voucher); }
        catch (\Throwable $e) { /* optional log */ }

        // Flash code for thanks page
        session()->flash('voucher_code', $voucher);

        if ($request->expectsJson() || $request->ajax()) {
            return response()->json([
                'ok'       => true,
                'redirect' => route('register.thanks'),
                'voucher'  => $voucher,
            ], 200);
        }

        return redirect()->route('register.thanks')->with('voucher_code', $voucher);
    }

    public function thanks()
    {
        return view('thanks', ['voucher' => session('voucher_code')]);
    }

    private function generateUniqueVoucher(): string
    {
        do {
            $raw  = strtoupper(Str::random(12));
            $base = preg_replace('/[^A-Z0-9]/', '', $raw);
            $base = substr($base, 0, 9);
            if (strlen($base) < 9) $base = strtoupper(str_pad($base, 9, 'X'));
            $code = substr($base, 0, 3) . '-' . substr($base, 3, 3) . '-' . substr($base, 6, 3);
        } while (Registration::where('voucher_code', $code)->exists());

        return $code;
    }



/**
 * Send WhatsApp template via DoubleTick (hard-coded creds).
 * V1 = mobile (E.164-ish), V2 = voucher.
 */
private function sendWhatsappTemplateHardcoded(string $mobileRaw, string $voucher): void
{
    // 1) Normalize "to" number:
    //    - If 10 digits → assume India and prefix 91
    //    - If starts with +, strip plus (API often expects digits only)
    //    - Strip all non-digits
    $digits = preg_replace('/\D/', '', $mobileRaw);
    if (strlen($digits) === 10) {
        $to = '91' . $digits;
    } elseif (str_starts_with($mobileRaw, '+')) {
        $to = ltrim($mobileRaw, '+');         // keep country code
        $to = preg_replace('/\D/', '', $to);  // digits only
    } else {
        $to = $digits; // already includes country code? leave as-is
    }

    // === Hard-coded credentials to match your CURL (change later to .env) ===
    $url       = 'https://public.doubletick.io/whatsapp/message/template';
    $authKey   = 'key_OzW3ateAEe'; // EXACTLY as in your curl
    $tplName   = 'registration_message_maybelline';
    $headerImg = 'https://data-storage.doubletick.io/org_Knlaeq1gF7/templates/683ad959-ede5-4eb1-85d1-d65d8e412e39.png';
    // ======================================================================

    $payload = [
        'messages' => [[
            'to'      => $to,
            'content' => [
                'templateName' => $tplName,
                'language'     => 'en',
                'templateData' => [
                    'header' => [
                        'type'     => 'IMAGE',
                        'mediaUrl' => $headerImg,
                        'filename' => basename(parse_url($headerImg, PHP_URL_PATH) ?? 'header.png'),
                    ],
                    'body' => [
                        // V1 = mobile, V2 = voucher (your request)
                        'placeholders' => [
                            $to,       // V1 (mobile to be displayed to user)
                            $voucher,  // V2 (voucher code)
                        ],
                    ],
                ],
            ],
        ]],
    ];

    Log::debug('DoubleTick REQUEST', ['to' => $to, 'payload' => $payload]);

    try {
        $res = Http::withHeaders([
                'accept'           => 'application/json',
                'content-type'     => 'application/json',
                'Authorization'    => $authKey,                 // not Bearer; exact key_...
                'X-Requested-With' => 'XMLHttpRequest',
            ])
            ->timeout(20)
            ->connectTimeout(10)
            ->withOptions(['http_errors' => false])
            ->post($url, $payload);

        $status = $res->status();
        $body   = json_decode($res->body(), true);
        if (!$res->successful()) {
            Log::warning('DoubleTick FAILED', ['status' => $status, 'body' => $body ?: $res->body()]);
        } else {
            Log::info('DoubleTick OK', ['status' => $status, 'body' => $body]);
        }
    } catch (\Throwable $e) {
        Log::error('DoubleTick EXCEPTION: '.$e->getMessage(), ['trace' => $e->getTraceAsString()]);
    }
}


    private function tryDecode(string $raw)
    {
        $j = json_decode($raw, true);
        return json_last_error() === JSON_ERROR_NONE ? $j : $raw;
    }

    // Helper to return a unified 422 response for file errors
    private function fileErrorResponse(Request $request, string $field, string $message)
    {
        if ($request->expectsJson() || $request->ajax()) {
            return response()->json(['ok' => false, 'errors' => [$field => [$message]]], 422);
        }
        return back()->withErrors([$field => $message])->withInput();
    }
}
