<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Maybelline Registration</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
	<link href="{{ asset('assets/css/style.css') }}" rel="stylesheet" type="text/css">
  <style>
    /* no design changes */
  </style>
</head>
<body>

<div class="container text-center py-4">

  <!-- Registration Form -->
  <div class="form-container">
    <!-- Logo Section -->
    <div class="logo" style="background-color: #000000;">
      <img src="{{ asset('assets/img/logo.png') }}" alt="Maybelline Logo" class="img-fluid">
    </div>

    <!-- Banner Section -->
    <div class="banner">
      <img src="{{ asset('assets/img/Asset-1ldpi.jpg') }}" alt="Maybelline Banner" class="img-fluid"> 
    </div>

    <h4 class="mb-3 text-center fw-bold">REGISTRATION FORM</h4>
    <p class="text-center mb-4">Please fill out the registration form below to get started</p>
    
    <!-- NOTE: id="regForm" + novalidate enables AJAX submit & inline errors without refresh -->
    <form id="regForm" action="{{ route('register.store') }}" method="POST" enctype="multipart/form-data" novalidate>
      @csrf

      <div class="mb-3">
        <input type="email" name="email" class="form-control" placeholder="Email ID" value="{{ old('email') }}" inputmode="email" autocomplete="email">
        <!-- inline error placeholder added by JS if needed -->
      </div>

      <div class="mb-3">
        <input
          type="tel"
          name="mobile"
          class="form-control"
          placeholder="Mobile Number"
          value="{{ old('mobile') }}"
          required
          inputmode="numeric"
          pattern="^\d{10}$"
          maxlength="10"
        >
        <!-- inline error placeholder added by JS if needed -->
      </div>

      <div class="mb-3">
        <select name="city" class="form-select" required>
          <option selected disabled>City</option>
          @foreach (['Delhi','Mumbai','Bangalore','Kolkata'] as $c)
            <option {{ old('city')===$c ? 'selected' : '' }}>{{ $c }}</option>
          @endforeach
        </select>
        <!-- inline error placeholder added by JS if needed -->
      </div>

      <div class="mb-3">
        <input
          type="text"
          name="invoice"
          class="form-control"
          placeholder="Invoice Value"
          value="{{ old('invoice') }}"
          required
          inputmode="numeric"
          pattern="^\d+$"
        >
        <!-- inline error placeholder added by JS if needed -->
      </div>

      <div class="mb-3">
        <div class="upload-box">
          <i class="bi bi-upload fs-3"></i>
          <p class="mb-0">Click to upload or drag and drop (image/PDF)</p>
          <input type="file" name="upload_file" class="form-control mt-2" accept="image/*,application/pdf" required>
        </div>
        <!-- inline error placeholder added by JS if needed -->
      </div>

      <div class="form-check mb-3">
        <input class="form-check-input" type="checkbox" id="terms" name="terms" {{ old('terms') ? 'checked' : '' }} required>
        <label class="form-check-label" for="terms">
          I agree to the terms and conditions
        </label>
        <!-- inline error placeholder added by JS if needed -->
      </div>

      <button type="submit" class="btn btn-dark w-50">SUBMIT</button>
    </form>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>

{{-- Digits-only + AJAX inline validation + robust AJAX submit (no visual change) --}}
<script>
(function(){
  const form     = document.getElementById('regForm');
  const csrf     = '{{ csrf_token() }}';
  const fields   = ['email','mobile','city','invoice','upload_file','terms'];

  // Force digits-only for Mobile & Invoice without changing design
  const mobileEl  = form.querySelector('[name="mobile"]');
  const invoiceEl = form.querySelector('[name="invoice"]');

  function digitsOnly(e){
    const before = e.target.value;
    const after  = before.replace(/\D+/g,'');
    if (before !== after) e.target.value = after;
  }
  if (mobileEl){
    mobileEl.setAttribute('pattern','^\\d{10}$');
    mobileEl.setAttribute('maxlength','10');
    mobileEl.setAttribute('inputmode','numeric');
    mobileEl.addEventListener('input', digitsOnly);
    mobileEl.addEventListener('keypress', e => { if (e.key && !/\d/.test(e.key)) e.preventDefault(); });
  }
  if (invoiceEl){
    invoiceEl.setAttribute('pattern','^\\d+$');
    invoiceEl.setAttribute('inputmode','numeric');
    invoiceEl.addEventListener('input', digitsOnly);
    invoiceEl.addEventListener('keypress', e => { if (e.key && !/\d/.test(e.key)) e.preventDefault(); });
  }

  // Error helpers (Bootstrap-compatible)
  function ensureErrorEl(input){
    let el = input.parentElement.querySelector('.invalid-feedback');
    if (!el) {
      el = document.createElement('div');
      el.className = 'invalid-feedback d-block';
      input.parentElement.appendChild(el);
    }
    return el;
  }
  function showError(input, message){
    input.classList.add('is-invalid');
    ensureErrorEl(input).textContent = message || 'Invalid';
  }
  function clearError(input){
    input.classList.remove('is-invalid');
    const el = input.parentElement.querySelector('.invalid-feedback');
    if (el) el.remove();
  }
  function clearAllErrors(){
    fields.forEach(name=>{
      const el = form.querySelector(`[name="${name}"]`);
      if (el) clearError(el);
    });
  }

  // Live AJAX field validation
  function debounce(fn, delay=300){ let t; return (...args)=>{ clearTimeout(t); t=setTimeout(()=>fn(...args), delay);} }
  async function validateField(name, value){
    try{
      const res = await fetch("{{ route('register.validate') }}", {
        method: 'POST',
        headers: {
          'Content-Type':'application/json',
          'X-CSRF-TOKEN': csrf,
          'Accept':'application/json',
          'X-Requested-With':'XMLHttpRequest'
        },
        body: JSON.stringify({ field: name, value: value })
      });
      if (res.status === 422){
        const data = await res.json();
        const msg = data?.errors?.[name]?.[0] || 'Invalid';
        return { ok:false, msg };
      }
      return { ok:true };
    } catch(err){ return { ok:true }; }
  }
  const handle = debounce(async (input)=>{
    const name = input.name;
    if (!fields.includes(name)) return;
    const value = (name==='terms') ? (input.checked ? '1' : '') : input.value.trim();

    // quick client-side checks to avoid round-trip
    if (name==='mobile' && !/^\d{10}$/.test(value)){
      showError(input, 'Enter a valid 10-digit mobile number.');
      return;
    }
    if (name==='invoice' && !/^\d+$/.test(value)){
      showError(input, 'Enter digits only.');
      return;
    }

    const { ok, msg } = await validateField(name, value);
    if (ok) clearError(input); else showError(input, msg);
  }, 250);

  document.addEventListener('DOMContentLoaded', ()=>{
    fields.forEach(n=>{
      const el = form.querySelector(`[name="${n}"]`);
      if (!el) return;
      el.addEventListener('input', ()=>handle(el));
      el.addEventListener('blur',  ()=>handle(el));
      if (n==='terms') el.addEventListener('change', ()=>handle(el));
    });
  });

  // AJAX submit (prevents refresh; shows inline errors) with robust error handling
  async function ajaxSubmit(e){
    e.preventDefault();
    clearAllErrors();

    // quick checks
    if (mobileEl && !/^\d{10}$/.test(mobileEl.value.trim())){
      showError(mobileEl, 'Enter a valid 10-digit mobile number.');
      mobileEl.focus(); return;
    }
    if (invoiceEl && !/^\d+$/.test(invoiceEl.value.trim())){
      showError(invoiceEl, 'Enter digits only.');
      invoiceEl.focus(); return;
    }

    const fd = new FormData(form); // includes _token and file

    try{
      const res = await fetch(form.action, {
        method: 'POST',
        headers: {
          // do NOT set Content-Type here for FormData
          'X-CSRF-TOKEN': csrf,
          'Accept': 'application/json',
          'X-Requested-With': 'XMLHttpRequest'
        },
        body: fd
      });

      // Validation errors
      if (res.status === 422){
        const data = await res.json();
        const errs = data?.errors || {};
        Object.keys(errs).forEach(name=>{
          const input = form.querySelector(`[name="${name}"]`);
          if (input) showError(input, errs[name][0]);
        });
        const first = form.querySelector('.is-invalid');
        if (first) first.focus();
        return; // stay on page
      }

      // CSRF expired or blocked
      if (res.status === 419){
        alert('Session expired. Please refresh the page and try again.');
        return;
      }

      // Payload too large
      if (res.status === 413){
        const up = form.querySelector('[name="upload_file"]');
        if (up) showError(up, 'File too large. Try a smaller image or PDF.');
        return;
      }

      // Other non-OK responses: try JSON, else text
      if (!res.ok){
        let msg = 'Something went wrong. Please try again.';
        try { const j = await res.json(); if (j?.message) msg = j.message; } catch(e) {
          try { msg = await res.text(); } catch(e2){}
        }
        alert(msg);
        return;
      }

      // Success path
      const data = await res.json();
      if (data.ok && data.redirect){
        window.location.href = data.redirect; // thanks page reads voucher from session
      } else {
        alert('Unexpected response. Please try again.');
      }

    } catch(err){
      alert('Network error. Please check your connection and try again.');
    }
  }
  form.addEventListener('submit', ajaxSubmit);
})();
</script>
</body>
</html>
